/* Direct3D Material
 * Copyright (c) 2002 Lionel ULMER
 * Copyright (c) 2006 Stefan DSINGER
 *
 * This file contains the implementation of Direct3DMaterial.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include "config.h"
#include "wine/port.h"
#include "wine/debug.h"

#include <assert.h>
#include <stdarg.h>
#include <string.h>
#include <stdlib.h>

#define COBJMACROS

#include "windef.h"
#include "winbase.h"
#include "winnls.h"
#include "winerror.h"
#include "wingdi.h"
#include "wine/exception.h"
#include "excpt.h"

#include "ddraw.h"
#include "d3d.h"

#include "ddraw_private.h"

WINE_DEFAULT_DEBUG_CHANNEL(d3d7);
WINE_DECLARE_DEBUG_CHANNEL(ddraw_thunk);

static void dump_material(LPD3DMATERIAL mat)
{
    DPRINTF("  dwSize : %ld\n", mat->dwSize);
}

/*****************************************************************************
 * IUnknown Methods.
 *****************************************************************************/

/*****************************************************************************
 * IDirect3DMaterial3::QueryInterface
 *
 * QueryInterface for IDirect3DMaterial. Can query all IDirect3DMaterial
 * versions.
 *
 * Params:
 *  riid: Interface id queried for
 *  obj: Address to pass the interface pointer back
 *
 * Returns:
 *  S_OK on success
 *  E_NOINTERFACE if the requested interface wasn't found
 *
 *****************************************************************************/
static HRESULT WINAPI
IDirect3DMaterialImpl_QueryInterface(IDirect3DMaterial3 *iface,
                                     REFIID riid,
                                     void **obj)
{
    ICOM_THIS_FROM(IDirect3DMaterialImpl, IDirect3DMaterial3, iface);
    TRACE("(%p)->(%s,%p)\n", This, debugstr_guid(riid), obj);

    *obj = NULL;

    if ( IsEqualGUID( &IID_IUnknown,  riid ) )
    {
        *obj = ICOM_INTERFACE(This, IDirect3DMaterial3);
        IDirect3DMaterial3_AddRef(ICOM_INTERFACE(This, IDirect3DMaterial3));
        TRACE("  Creating IUnknown interface at %p.\n", *obj);
        return S_OK;
    }
    if ( IsEqualGUID( &IID_IDirect3DMaterial, riid ) )
    {
        *obj = ICOM_INTERFACE(This, IDirect3DMaterial);
        IDirect3DMaterial3_AddRef(ICOM_INTERFACE(This, IDirect3DMaterial3));
        TRACE("  Creating IDirect3DMaterial interface %p\n", *obj);
        return S_OK;
    }
    if ( IsEqualGUID( &IID_IDirect3DMaterial2, riid ) )
    {
        *obj = ICOM_INTERFACE(This, IDirect3DMaterial2);
        IDirect3DMaterial3_AddRef(ICOM_INTERFACE(This, IDirect3DMaterial3));
        TRACE("  Creating IDirect3DMaterial2 interface %p\n", *obj);
        return S_OK;
    }
    if ( IsEqualGUID( &IID_IDirect3DMaterial3, riid ) )
    {
        *obj = ICOM_INTERFACE(This, IDirect3DMaterial3);
        IDirect3DMaterial3_AddRef(ICOM_INTERFACE(This, IDirect3DMaterial3));
        TRACE("  Creating IDirect3DMaterial3 interface %p\n", *obj);
        return S_OK;
    }

    FIXME("(%p): interface for IID %s NOT found!\n", This, debugstr_guid(riid));
    return E_NOINTERFACE;
}

static HRESULT WINAPI
Thunk_IDirect3DMaterialImpl_2_QueryInterface(IDirect3DMaterial2 *iface,
                                            REFIID riid,
                                            void **obj)
{
    ICOM_THIS_FROM(IDirect3DMaterialImpl, IDirect3DMaterial2, iface);
    TRACE_(ddraw_thunk)("(%p)->(%s,%p) thunking to IDirect3DMaterial3 interface.\n", This, debugstr_guid(riid), obj);
    return IDirect3DMaterial3_QueryInterface(ICOM_INTERFACE(This, IDirect3DMaterial3),
                                             riid,
                                             obj);
}

static HRESULT WINAPI
Thunk_IDirect3DMaterialImpl_1_QueryInterface(IDirect3DMaterial *iface,
                                            REFIID riid, 
                                            void **obj)
{
    ICOM_THIS_FROM(IDirect3DMaterialImpl, IDirect3DMaterial, iface);
    TRACE_(ddraw_thunk)("(%p)->(%s,%p) thunking to IDirect3DMaterial3 interface.\n", This, debugstr_guid(riid), obj);
    return IDirect3DMaterial3_QueryInterface(ICOM_INTERFACE(This, IDirect3DMaterial3),
                                             riid,
                                             obj);
}

/*****************************************************************************
 * IDirect3DMaterial3::AddRef
 *
 * Increases the refcount.
 *
 * Returns:
 *  The new refcount
 *
 *****************************************************************************/
static ULONG WINAPI
IDirect3DMaterialImpl_AddRef(IDirect3DMaterial3 *iface)
{
    ICOM_THIS_FROM(IDirect3DMaterialImpl, IDirect3DMaterial3, iface);
    ULONG ref = InterlockedIncrement(&This->ref);

    TRACE("(%p)->() incrementing from %lu.\n", This, ref - 1);

    return ref;
}

static ULONG WINAPI
Thunk_IDirect3DMaterialImpl_2_AddRef(IDirect3DMaterial2 *iface)
{
    ICOM_THIS_FROM(IDirect3DMaterialImpl, IDirect3DMaterial2, iface);
    TRACE_(ddraw_thunk)("(%p)->() thunking to IDirect3DMaterial3 interface.\n", This);
    return IDirect3DMaterial3_AddRef(ICOM_INTERFACE(This, IDirect3DMaterial3));
}

static ULONG WINAPI
Thunk_IDirect3DMaterialImpl_1_AddRef(IDirect3DMaterial *iface)
{
    ICOM_THIS_FROM(IDirect3DMaterialImpl, IDirect3DMaterial, iface);
    TRACE_(ddraw_thunk)("(%p)->() thunking to IDirect3DMaterial3 interface.\n", iface);
    return IDirect3DMaterial3_AddRef(ICOM_INTERFACE(This, IDirect3DMaterial3));
}

/*****************************************************************************
 * IDirect3DMaterial3::Release
 *
 * Reduces the refcount by one. If the refcount falls to 0, the object
 * is destroyed
 *
 * Returns:
 *  The new refcount
 *
 *****************************************************************************/
static ULONG WINAPI
IDirect3DMaterialImpl_Release(IDirect3DMaterial3 *iface)
{
    ICOM_THIS_FROM(IDirect3DMaterialImpl, IDirect3DMaterial3, iface);
    ULONG ref = InterlockedDecrement(&This->ref);

    TRACE("(%p)->() decrementing from %lu.\n", This, ref + 1);

    if (!ref)
    {
        HeapFree(GetProcessHeap(), 0, This);
        return 0;
    }
    return ref;
}

static ULONG WINAPI
Thunk_IDirect3DMaterialImpl_2_Release(IDirect3DMaterial2 *iface)
{
    ICOM_THIS_FROM(IDirect3DMaterialImpl, IDirect3DMaterial2, iface);
    TRACE_(ddraw_thunk)("(%p)->() thunking to IDirect3DMaterial3 interface.\n", This);
    return IDirect3DMaterial3_Release(ICOM_INTERFACE(This, IDirect3DMaterial3));
}

static ULONG WINAPI
Thunk_IDirect3DMaterialImpl_1_Release(IDirect3DMaterial *iface)
{
    ICOM_THIS_FROM(IDirect3DMaterialImpl, IDirect3DMaterial, iface);
    TRACE_(ddraw_thunk)("(%p)->() thunking to IDirect3DMaterial3 interface.\n", This);
    return IDirect3DMaterial3_Release(ICOM_INTERFACE(This, IDirect3DMaterial3));
}

/*****************************************************************************
 * IDirect3DMaterial Methods
 *****************************************************************************/

/*****************************************************************************
 * IDirect3DMaterial::Initialize
 *
 * A no-op initialization
 *
 * Params:
 *  Direct3D: Pointer to a Direct3D interface
 *
 * Returns:
 *  D3D_OK
 *
 *****************************************************************************/
static HRESULT WINAPI
IDirect3DMaterialImpl1_Initialize(IDirect3DMaterial *iface,
                                  IDirect3D *Direct3D)
{
    ICOM_THIS_FROM(IDirect3DMaterialImpl, IDirect3DMaterial, iface);

    TRACE("(%p)->(%p) no-op...!\n", This, Direct3D);

    return D3D_OK;
}

/*****************************************************************************
 * IDirect3DMaterial::Reserve
 *
 * DirectX 5 sdk: "The IDirect3DMaterial2::Reserve method is not implemented"
 * Odd. They seem to have mixed their interfaces.
 *
 * Returns:
 *  DDERR_UNSUPPORTED
 *
 *****************************************************************************/
static HRESULT WINAPI
IDirect3DMaterialImpl_Reserve(IDirect3DMaterial *iface)
{
    ICOM_THIS_FROM(IDirect3DMaterialImpl, IDirect3DMaterial, iface);
    TRACE("(%p)->() not implemented\n", This);

    return DDERR_UNSUPPORTED;
}

/*****************************************************************************
 * IDirect3DMaterial::Unreserve
 *
 * Not supported too
 *
 * Returns:
 *  DDERR_UNSUPPORTED
 *
 *****************************************************************************/
static HRESULT WINAPI
IDirect3DMaterialImpl_Unreserve(IDirect3DMaterial *iface)
{
    ICOM_THIS_FROM(IDirect3DMaterialImpl, IDirect3DMaterial, iface);
    TRACE("(%p)->() not implemented.\n", This);

    return DDERR_UNSUPPORTED;
}

/*****************************************************************************
 * IDirect3DMaterial3::SetMaterial
 *
 * Sets the material description
 *
 * Params:
 *  Mat: Material to set
 *
 * Returns:
 *  D3D_OK on success
 *  DDERR_INVALIDPARAMS if Mat is NULL
 *
 *****************************************************************************/
static HRESULT WINAPI
IDirect3DMaterialImpl_SetMaterial(IDirect3DMaterial3 *iface,
                                  D3DMATERIAL *Mat)
{
    ICOM_THIS_FROM(IDirect3DMaterialImpl, IDirect3DMaterial3, iface);
    TRACE("(%p)->(%p)\n", This, Mat);

    if(!Mat)
        return DDERR_INVALIDPARAMS;

    if (TRACE_ON(d3d7))
        dump_material(Mat);

    /* Stores the material */
    memset(&This->mat, 0, sizeof(This->mat));
    memcpy(&This->mat, Mat, Mat->dwSize);

    return D3D_OK;
}

static HRESULT WINAPI
Thunk_IDirect3DMaterialImpl_2_SetMaterial(IDirect3DMaterial2 *iface,
                                         D3DMATERIAL *Mat)
{
    ICOM_THIS_FROM(IDirect3DMaterialImpl, IDirect3DMaterial2, iface);
    TRACE_(ddraw_thunk)("(%p)->(%p) thunking to IDirect3DMaterial3 interface.\n", This, Mat);
    return IDirect3DMaterial3_SetMaterial(ICOM_INTERFACE(This, IDirect3DMaterial3),
                                          Mat);
}

static HRESULT WINAPI
Thunk_IDirect3DMaterialImpl_1_SetMaterial(IDirect3DMaterial *iface,
                                         D3DMATERIAL *Mat)
{
    ICOM_THIS_FROM(IDirect3DMaterialImpl, IDirect3DMaterial, iface);
    TRACE_(ddraw_thunk)("(%p)->(%p) thunking to IDirect3DMaterial3 interface.\n", This, Mat);
    return IDirect3DMaterial3_SetMaterial(ICOM_INTERFACE(This, IDirect3DMaterial3),
                                          Mat);
}

/*****************************************************************************
 * IDirect3DMaterial3::GetMaterial
 *
 * Returns the material assigned to this interface
 *
 * Params:
 *  Mat: Pointer to a D3DMATERIAL structure to store the material description
 *
 * Returns:
 *  D3D_OK on success
 *  DDERR_INVALIDPARAMS if Mat is NULL
 *
 *****************************************************************************/
static HRESULT WINAPI
IDirect3DMaterialImpl_GetMaterial(IDirect3DMaterial3 *iface,
                                  D3DMATERIAL *Mat)
{
    ICOM_THIS_FROM(IDirect3DMaterialImpl, IDirect3DMaterial3, iface);
    DWORD Size;
    TRACE("(%p/%p)->(%p)\n", This, iface, Mat);

    if(!Mat)
        return DDERR_INVALIDPARAMS;

    if (TRACE_ON(d3d7))
    {
        TRACE("  Returning material : ");
        dump_material(&This->mat);
    }

    /* Copies the material structure */
    Size = Mat->dwSize;
    memset(Mat, 0, Size);
    memcpy(Mat, &This->mat, Size);

    return DD_OK;
}

static HRESULT WINAPI
Thunk_IDirect3DMaterialImpl_2_GetMaterial(IDirect3DMaterial2 *iface,
                                         D3DMATERIAL *Mat)
{
    ICOM_THIS_FROM(IDirect3DMaterialImpl, IDirect3DMaterial2, iface);
    TRACE("(%p)->(%p) thunking to IDirect3DMaterial3 interface.\n", This, Mat);
    return IDirect3DMaterial3_GetMaterial(ICOM_INTERFACE(This, IDirect3DMaterial3),
                                          Mat);
}

static HRESULT WINAPI
Thunk_IDirect3DMaterialImpl_1_GetMaterial(IDirect3DMaterial *iface,
                                         D3DMATERIAL *Mat)
{
    ICOM_THIS_FROM(IDirect3DMaterialImpl, IDirect3DMaterial, iface);
    TRACE("(%p)->(%p) thunking to IDirect3DMaterial3 interface.\n", This, Mat);
    return IDirect3DMaterial3_GetMaterial(ICOM_INTERFACE(This, IDirect3DMaterial3),
                                          Mat);
}

/*****************************************************************************
 * IDirect3DMaterial3::GetHandle
 *
 * Returns a handle for the material interface. The handle is simply a
 * pointer to the material implementation
 *
 * Params:
 *  Direct3DDevice3: The device this handle is assigned to
 *  Handle: Address to write the handle to
 *
 * Returns:
 *  D3D_OK on success
 *  DDERR_INVALIDPARAMS if Handle is NULL
 *
 *****************************************************************************/
static HRESULT WINAPI
IDirect3DMaterialImpl_GetHandle(IDirect3DMaterial3 *iface,
                                IDirect3DDevice3 *Direct3DDevice3,
                                D3DMATERIALHANDLE *Handle)
{
    ICOM_THIS_FROM(IDirect3DMaterialImpl, IDirect3DMaterial3, iface);
    TRACE("(%p)->(%p,%p)\n", This, Direct3DDevice3, Handle);

    if(!Handle)
        return DDERR_INVALIDPARAMS;

    This->active_device = (IDirect3DDeviceImpl *) Direct3DDevice3;
    *Handle = (DWORD) This; /* Warning: this is not 64 bit clean.
                             * Maybe also we need to store this
                             * material somewhere in the device ?
                             */

    TRACE(" returning handle %08lx.\n", *Handle);

    return DD_OK;
}

static HRESULT WINAPI
Thunk_IDirect3DMaterialImpl_2_GetHandle(IDirect3DMaterial2 *iface,
                                        IDirect3DDevice2 *Direct3DDevice2,
                                        D3DMATERIALHANDLE *Handle)
{
    ICOM_THIS_FROM(IDirect3DMaterialImpl, IDirect3DMaterial2, iface);
    IDirect3DDeviceImpl *device = ICOM_OBJECT(IDirect3DDeviceImpl, IDirect3DDevice2, Direct3DDevice2);
    TRACE("(%p)->(%p,%p) thunking to IDirect3DMaterial3 interface.\n", This, device, Handle);
    return IDirect3DMaterial3_GetHandle(ICOM_INTERFACE(This, IDirect3DMaterial3),
                                        ICOM_INTERFACE(device, IDirect3DDevice3),
                                        Handle);
}

static HRESULT WINAPI
Thunk_IDirect3DMaterialImpl_1_GetHandle(IDirect3DMaterial *iface,
                                        IDirect3DDevice *Direct3DDevice,
                                        D3DMATERIALHANDLE *Handle)
{
    ICOM_THIS_FROM(IDirect3DMaterialImpl, IDirect3DMaterial, iface);
    IDirect3DDeviceImpl *device = ICOM_OBJECT(IDirect3DDeviceImpl, IDirect3DDevice, Direct3DDevice);
    TRACE("(%p)->(%p,%p) thunking to IDirect3DMaterial3 interface.\n", This, device, Handle);
    return IDirect3DMaterial3_GetHandle(ICOM_INTERFACE(This, IDirect3DMaterial3),
                                        ICOM_INTERFACE(device, IDirect3DDevice3),
                                        Handle);
}

/*****************************************************************************
 * material_activate
 *
 * Uses IDirect3DDevice7::SetMaterial to activate the material
 *
 * Params:
 *  This: Pointer to the material implementation to activate
 *
 *****************************************************************************/
void material_activate(IDirect3DMaterialImpl* This)
{
    D3DMATERIAL7 d3d7mat;

    TRACE("Activating material %p\n", This);
    d3d7mat.diffuse = This->mat.diffuse;
    d3d7mat.ambient = This->mat.ambient;
    d3d7mat.specular = This->mat.specular;
    d3d7mat.emissive = This->mat.emissive;
    d3d7mat.power = This->mat.power;

    IDirect3DDevice7_SetMaterial(ICOM_INTERFACE(This->active_device, IDirect3DDevice7),
                                 &d3d7mat);
}

const IDirect3DMaterial3Vtbl IDirect3DMaterial3_Vtbl =
{
    /*** IUnknown Methods ***/
    IDirect3DMaterialImpl_QueryInterface,
    IDirect3DMaterialImpl_AddRef,
    IDirect3DMaterialImpl_Release,
    /*** IDirect3DMaterial3 Methods ***/
    IDirect3DMaterialImpl_SetMaterial,
    IDirect3DMaterialImpl_GetMaterial,
    IDirect3DMaterialImpl_GetHandle,
};

const IDirect3DMaterial2Vtbl IDirect3DMaterial2_Vtbl =
{
    /*** IUnknown Methods ***/
    Thunk_IDirect3DMaterialImpl_2_QueryInterface,
    Thunk_IDirect3DMaterialImpl_2_AddRef,
    Thunk_IDirect3DMaterialImpl_2_Release,
    /*** IDirect3DMaterial2 Methods ***/
    Thunk_IDirect3DMaterialImpl_2_SetMaterial,
    Thunk_IDirect3DMaterialImpl_2_GetMaterial,
    Thunk_IDirect3DMaterialImpl_2_GetHandle,
};

const IDirect3DMaterialVtbl IDirect3DMaterial_Vtbl =
{
    /*** IUnknown Methods ***/
    Thunk_IDirect3DMaterialImpl_1_QueryInterface,
    Thunk_IDirect3DMaterialImpl_1_AddRef,
    Thunk_IDirect3DMaterialImpl_1_Release,
    /*** IDirect3DMaterial1 Methods ***/
    IDirect3DMaterialImpl1_Initialize,
    Thunk_IDirect3DMaterialImpl_1_SetMaterial,
    Thunk_IDirect3DMaterialImpl_1_GetMaterial,
    Thunk_IDirect3DMaterialImpl_1_GetHandle,
    IDirect3DMaterialImpl_Reserve,
    IDirect3DMaterialImpl_Unreserve
};
