/*
 * Test for IDirect3DVertexBuffer7::ProcessVertices
 *
 * Copyright (C) 2005 Antoine Chavasse
 * Copyright (C) 2006 Stefan Dsinger
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <assert.h>
#include "wine/test.h"
#include "ddraw.h"
#include "d3d.h"

#include <stdio.h>

static LPDIRECTDRAW7           lpDD = NULL;
static LPDIRECT3D7             lpD3D = NULL;
static LPDIRECTDRAWSURFACE7    lpDDS = NULL;
static LPDIRECT3DDEVICE7       lpD3DDevice = NULL;
static LPDIRECT3DVERTEXBUFFER7 lpVBuf1 = NULL;
static LPDIRECT3DVERTEXBUFFER7 lpVBuf2 = NULL;

static HRESULT (WINAPI *pDirectDrawCreateEx)(LPGUID,LPVOID*,REFIID,LPUNKNOWN);

typedef struct _VERTEX
{
    float x, y, z;// position
    DWORD diffuse;
    float tu1, tv1;    // Texture
//     float tu2, tv2;    // Texture
} VERTEX, *LPVERTEX; 

typedef struct _RHWVERTEX
{
    float x, y, z;// position
    float rhw;
    DWORD diffuse;
    float tu1, tv1;    // Texture
//     float tu2, tv2;    // Texture
} RHWVERTEX, *LPRHWVERTEX; 

static void init_function_pointers(void)
{
    HMODULE hmod = GetModuleHandleA("ddraw.dll");

    if(hmod)
    {
        pDirectDrawCreateEx = (void*)GetProcAddress(hmod, "DirectDrawCreateEx");
    }
}


static BOOL CreateDirect3D(void)
{
    HRESULT rc;
    DDSURFACEDESC2 ddsd;
    D3DVERTEXBUFFERDESC VertexBufferDesc;

    rc = pDirectDrawCreateEx(NULL, (void**)&lpDD,
        &IID_IDirectDraw7, NULL);
    ok(rc==DD_OK || rc==DDERR_NODIRECTDRAWSUPPORT, "DirectDrawCreateEx returned: %lx\n", rc);
    if (!lpDD) {
        trace("DirectDrawCreateEx() failed with an error %lx\n", rc);
        return FALSE;
    }

    rc = IDirectDraw_SetCooperativeLevel(lpDD, NULL, DDSCL_NORMAL);
    ok(rc==DD_OK, "SetCooperativeLevel returned: %lx\n", rc);

    rc = IDirectDraw7_QueryInterface(lpDD, &IID_IDirect3D7, (void**) &lpD3D);
    ok(rc==DD_OK, "QueryInterface returned: %lx\n", rc);

    memset(&ddsd, 0, sizeof(ddsd));
    ddsd.dwSize = sizeof(ddsd);
    ddsd.dwFlags = DDSD_CAPS | DDSD_WIDTH | DDSD_HEIGHT;
    ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN | DDSCAPS_3DDEVICE;
    ddsd.dwWidth = 640;
    ddsd.dwHeight = 480;
    rc = IDirectDraw7_CreateSurface(lpDD, &ddsd, &lpDDS, NULL);
    ok(rc==DD_OK, "CreateSurface returned: %lx\n", rc);

    rc = IDirect3D7_CreateDevice(lpD3D, &IID_IDirect3DHALDevice, lpDDS,
        &lpD3DDevice);
    ok(rc==D3D_OK || rc==DDERR_NOPALETTEATTACHED || rc==E_OUTOFMEMORY, "CreateDevice returned: %lx\n", rc);
    if (!lpD3DDevice) {
        trace("IDirect3D7::CreateDevice() failed with an error %lx\n", rc);
        return FALSE;
    }

    memset(&VertexBufferDesc, 0, sizeof(VertexBufferDesc));
    VertexBufferDesc.dwSize = sizeof(VertexBufferDesc);
    VertexBufferDesc.dwCaps = 0;
    VertexBufferDesc.dwFVF = D3DFVF_XYZ | D3DFVF_DIFFUSE | D3DFVF_TEX1;
    VertexBufferDesc.dwNumVertices = 20;

    rc = IDirect3D7_CreateVertexBuffer(lpD3D, &VertexBufferDesc, &lpVBuf1, 0);
    ok(rc==D3D_OK || rc==E_OUTOFMEMORY, "CreateVertexBuffer returned: %lx\n", rc);
    if (!lpVBuf1) {
        trace("IDirect3D7::CreateVertexBuffer() failed with an error %lx\n", rc);
        return FALSE;
    }

    memset(&VertexBufferDesc, 0, sizeof(VertexBufferDesc));
    VertexBufferDesc.dwSize = sizeof(VertexBufferDesc);
    VertexBufferDesc.dwCaps = 0;
    VertexBufferDesc.dwFVF = D3DFVF_XYZRHW | D3DFVF_DIFFUSE | D3DFVF_TEX1;
    VertexBufferDesc.dwNumVertices = 20;

    rc = IDirect3D7_CreateVertexBuffer(lpD3D, &VertexBufferDesc, &lpVBuf2, 4);
    ok(rc==D3D_OK || rc==E_OUTOFMEMORY, "CreateVertexBuffer returned: %lx\n", rc);
    if (!lpVBuf2) {
        trace("IDirect3D7::CreateVertexBuffer() failed with an error %lx\n", rc);
        return FALSE;
    }

    return TRUE;
}

static void ReleaseDirect3D(void)
{
    if (lpVBuf2 != NULL)
    {
        IDirect3DVertexBuffer7_Release(lpVBuf2);
        lpVBuf2 = NULL;
    }
    if (lpVBuf1 != NULL)
    {
        IDirect3DVertexBuffer7_Release(lpVBuf1);
        lpVBuf1 = NULL;
    }
    if (lpD3DDevice != NULL)
    {
        IDirect3DDevice7_Release(lpD3DDevice);
        lpD3DDevice = NULL;
    }

    if (lpDDS != NULL)
    {
        IDirectDrawSurface_Release(lpDDS);
        lpDDS = NULL;
    }

    if (lpD3D != NULL)
    {
        IDirect3D7_Release(lpD3D);
        lpD3D = NULL;
    }

    if (lpDD != NULL)
    {
        IDirectDraw_Release(lpDD);
        lpDD = NULL;
    }
}

static void ProcessTest(void)
{
    VERTEX *data1 = NULL;
    RHWVERTEX *data2 = NULL;
    HRESULT rc;
    int i;
    D3DVIEWPORT7 vp;
    DWORD count;
    D3DMATRIX view = { 1.0, 0.0, 0.0, 0.0,
                       0.0, 1.0, 0.0, 0.0,
                       0.0, 0.0, 1.0, 0.0,
                       0.0, 0.0, 0.0, 1.0 };

    D3DMATRIX world = {  1.000000,    0.000000,   0.000000,  0.000000,
                         0.000000,    1.000000,   0.000000,  0.000000,
                         0.000000,    0.000000,   1.000000,  0.000000,
                         0.000000,    0.000000,   0.000000,  1.000000 };

    D3DMATRIX proj = {  1.000000,  0.000000,  0.000000,  0.000000,
                        0.000000,  1.000000,  0.000000,  0.000000,
                        0.000000,  0.000000,  1.000000,  0.000000,
                        0.000000,  0.000000,  0.000000,  1.000000 };

//     proj._22 = (float) 1 + (float) 1 / (float) 3;

    /* Check the default viewport */
    rc = IDirect3DDevice7_GetViewport(lpD3DDevice, &vp);
    printf("vp.dwX = %ld\n", vp.dwX);
    printf("vp.dwY = %ld\n", vp.dwY);
    printf("vp.dwWidth = %ld\n", vp.dwWidth);
    printf("vp.dwHeight = %ld\n", vp.dwHeight);
    printf("vp.dvMinZ = %f\n", vp.dvMinZ);
    printf("vp.dvMaxZ = %f\n", vp.dvMaxZ);

    vp.dwX = 0;
    vp.dwY = 0;
    vp.dwWidth = 256;
    vp.dwHeight = 256;
    vp.dvMinZ = 0.0;
    vp.dvMaxZ = 1.0;
//     rc = IDirect3DDevice7_SetViewport(lpD3DDevice, &vp);
//     ok(rc==D3D_OK, "IDirect3DDevice7_SetViewport failed");

    rc = IDirect3DDevice7_SetTransform(lpD3DDevice, D3DTRANSFORMSTATE_VIEW, &view);
    ok(rc==D3D_OK, "IDirect3DDevice7_SetTransform failed");

    rc = IDirect3DDevice7_SetTransform(lpD3DDevice, D3DTRANSFORMSTATE_PROJECTION, &proj);
    ok(rc==D3D_OK, "IDirect3DDevice7_SetTransform failed");

    rc = IDirect3DDevice7_SetTransform(lpD3DDevice, D3DTRANSFORMSTATE_WORLD, &world);
    ok(rc==D3D_OK, "IDirect3DDevice7_SetTransform failed");

    rc = IDirect3DVertexBuffer7_Lock(lpVBuf1, 0, (void **) &data1, NULL);
    ok(rc == D3D_OK, "IDirect3DVertexBuffer7::Lock returned %lx\n", rc);
    if(data1 == NULL)
    {
        trace("IDirect3DVertexBuffer7::Lock failed with error %lx\n", rc);
        return;
    }

    memset(data1, 0, sizeof(VERTEX) * 20);

    for(i=0; i < 20; i++)
    {
        data1[i].diffuse = 0x12345678;
        data1[i].tu1 = 1.0;
        data1[i].tv1 = 1.5;
//         data1[i].tu2 = 2.0;
//         data1[i].tv2 = 2.5;
    }

    count=12;

    data1[0].x =  0.0;
    data1[0].y =  0.0;
    data1[0].z =  0.0;

    data1[1].x =  1.0;
    data1[1].y =  1.0;
    data1[1].z =  0.0;

    data1[2].x =  -1.0;
    data1[2].y =  -1.0;
    data1[2].z =  0.0;

    data1[3].x =  0.5;
    data1[3].y =  -0.5;
    data1[3].z =  0.0;

    data1[4].x =  0.0;
    data1[4].y =  0.0;
    data1[4].z =  0.0;

    data1[5].x =  0.0;
    data1[5].y =  0.0;
    data1[5].z =  0.0;

    data1[6].x =  0.0;
    data1[6].y =  0.0;
    data1[6].z =  0.0;

    data1[7].x =  0.0;
    data1[7].y =  0.0;
    data1[7].z =  0.0;

    data1[8].x =  0.0;
    data1[8].y =  0.0;
    data1[8].z =  0.0;

    data1[9].x =  0.0;
    data1[9].y =  0.0;
    data1[9].z =  0.0;

    data1[10].x =  0.0;
    data1[10].y =  0.0;
    data1[10].z =  0.0;

    data1[11].x =  0.0;
    data1[11].y =  0.0;
    data1[11].z =  0.0;

    rc = IDirect3DVertexBuffer7_Unlock(lpVBuf1);
    ok(rc == D3D_OK, "IDirect3DVertexBuffer7::Unlock returned %lx\n", rc);

    rc = IDirect3DVertexBuffer7_ProcessVertices(lpVBuf2, D3DVOP_TRANSFORM | D3DVOP_CLIP, 0, count, lpVBuf1, 0, lpD3DDevice, 0);
    ok(rc == D3D_OK, "IDirect3DVertexBuffer7::ProcessVertices returned %lx\n", rc);
    if(rc != D3D_OK) return;

    rc = IDirect3DVertexBuffer7_Lock(lpVBuf2, 0, (void **) &data2, NULL);
    ok(rc == D3D_OK, "IDirect3DVertexBuffer7::Lock returned %lx\n", rc);
    if(data2 == NULL)
    {
        trace("IDirect3DVertexBuffer7::Lock failed with error %lx\n", rc);
        return;
    }

    printf("Dumping resulting vertices\n");
    for(i=0; i < 20; i++)
    {
        printf("Vertex %02d: ( %06.2f  %06.2f  %06.2f ) rhw = %06.2f\n", i, data2[i].x, data2[i].y, data2[i].z, data2[i].rhw);
        
        if(i < count)
        {
            if(data2[i].diffuse != 0x12345678) printf("Diffuse is %08lx\n", data2[i].diffuse);
            if(data2[i].tu1 != 1.0) printf("tu1 is %f\n", data2[i].tu1);
            if(data2[i].tv1 != 1.5) printf("tv1 is %f\n", data2[i].tv1);
//             if(data2[i].tu2 != 2.0) printf("tu2 is %f\n", data2[i].tu2);
//             if(data2[i].tv2 != 2.5) printf("tv2 is %f\n", data2[i].tv2);
        }
    }

    rc = IDirect3DVertexBuffer7_Unlock(lpVBuf2);
    ok(rc == D3D_OK, "IDirect3DVertexBuffer7::Unlock returned %lx\n", rc);
}

START_TEST(processvertices)
{
    init_function_pointers();
    if(!pDirectDrawCreateEx) {
        trace("function DirectDrawCreateEx not available, skipping tests\n");
        return;
    }

    if(!CreateDirect3D()) {
        trace("Skipping tests\n");
        return;
	}
    ProcessTest();
    ReleaseDirect3D();
}
