/* Direct3D Viewport
 * Copyright (c) 1998 Lionel ULMER
 * Copyright (c) 2006 Stefan DSINGER
 *
 * This file contains the implementation of Direct3DViewport2.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include "config.h"
#include "wine/port.h"
#include "wine/debug.h"

#include <assert.h>
#include <stdarg.h>
#include <string.h>
#include <stdlib.h>

#define COBJMACROS

#include "windef.h"
#include "winbase.h"
#include "winnls.h"
#include "winerror.h"
#include "wingdi.h"
#include "wine/exception.h"
#include "excpt.h"

#include "ddraw.h"
#include "d3d.h"

#include "ddraw_private.h"

WINE_DEFAULT_DEBUG_CHANNEL(d3d7);

/*****************************************************************************
 * Helper functions
 *****************************************************************************/

/*****************************************************************************
 * viewport_activate
 *
 * activates the viewport using IDirect3DDevice7::SetViewport
 *
 *****************************************************************************/
void viewport_activate(IDirect3DViewportImpl* This)
{
    IDirect3DLightImpl* light;
    D3DVIEWPORT7 vp;

    /* Activate all the lights associated with this context */
    light = This->lights;

    while (light != NULL)
    {
        light->activate(light);
        light = light->next;
    }

    /* And copy the values in the structure used by the device */
    if (This->use_vp2)
    {
        vp.dwX = This->viewports.vp2.dwX;
        vp.dwY = This->viewports.vp2.dwY;
        vp.dwHeight = This->viewports.vp2.dwHeight;
        vp.dwWidth = This->viewports.vp2.dwWidth;
        vp.dvMinZ = This->viewports.vp2.dvMinZ;
        vp.dvMaxZ = This->viewports.vp2.dvMaxZ;
    }
    else
    {
        vp.dwX = This->viewports.vp1.dwX;
        vp.dwY = This->viewports.vp1.dwY;
        vp.dwHeight = This->viewports.vp1.dwHeight;
        vp.dwWidth = This->viewports.vp1.dwWidth;
        vp.dvMinZ = This->viewports.vp1.dvMinZ;
        vp.dvMaxZ = This->viewports.vp1.dvMaxZ;
    }

    /* And also set the viewport */
    IDirect3DDevice7_SetViewport(ICOM_INTERFACE(This->active_device, IDirect3DDevice7),
                                &vp);
}

/*****************************************************************************
 * _dump_D3DVIEWPORT, _dump_D3DVIEWPORT2
 *
 * Writes viewport information to TRACE
 *
 *****************************************************************************/
static void _dump_D3DVIEWPORT(D3DVIEWPORT *lpvp)
{
    TRACE("    - dwSize = %ld   dwX = %ld   dwY = %ld\n",
          lpvp->dwSize, lpvp->dwX, lpvp->dwY);
    TRACE("    - dwWidth = %ld   dwHeight = %ld\n",
          lpvp->dwWidth, lpvp->dwHeight);
    TRACE("    - dvScaleX = %f   dvScaleY = %f\n",
          lpvp->dvScaleX, lpvp->dvScaleY);
    TRACE("    - dvMaxX = %f   dvMaxY = %f\n",
          lpvp->dvMaxX, lpvp->dvMaxY);
    TRACE("    - dvMinZ = %f   dvMaxZ = %f\n",
          lpvp->dvMinZ, lpvp->dvMaxZ);
}

static void _dump_D3DVIEWPORT2(D3DVIEWPORT2 *lpvp)
{
    TRACE("    - dwSize = %ld   dwX = %ld   dwY = %ld\n",
          lpvp->dwSize, lpvp->dwX, lpvp->dwY);
    TRACE("    - dwWidth = %ld   dwHeight = %ld\n",
          lpvp->dwWidth, lpvp->dwHeight);
    TRACE("    - dvClipX = %f   dvClipY = %f\n",
          lpvp->dvClipX, lpvp->dvClipY);
    TRACE("    - dvClipWidth = %f   dvClipHeight = %f\n",
          lpvp->dvClipWidth, lpvp->dvClipHeight);
    TRACE("    - dvMinZ = %f   dvMaxZ = %f\n",
          lpvp->dvMinZ, lpvp->dvMaxZ);
}

/*****************************************************************************
 * IUnknown Methods.
 *****************************************************************************/

/*****************************************************************************
 * IDirect3DViewport3::QueryInterface
 *
 * A normal QueryInterface. Can query all interface versions and the
 * IUnknown interface. The VTables of the different versions
 * are equal
 *
 * Params:
 *  refiid: Interface id queried for
 *  obj: Address to write the interface pointer to
 *
 * Returns:
 *  S_OK on success.
 *  E_NOINTERFACE if the requested interface wasn't found
 *
 *****************************************************************************/
static HRESULT WINAPI
IDirect3DViewportImpl_QueryInterface(IDirect3DViewport3 *iface,
                                     REFIID refiid,
                                     void **obj)
{
    ICOM_THIS_FROM(IDirect3DViewportImpl, IDirect3DViewport3, iface);
    TRACE("(%p)->(%s,%p)\n", This, debugstr_guid(refiid), obj);

    *obj = NULL;

    if ( IsEqualGUID(&IID_IUnknown,  refiid) ||
         IsEqualGUID(&IID_IDirect3DViewport, refiid) ||
         IsEqualGUID(&IID_IDirect3DViewport2, refiid) ||
         IsEqualGUID(&IID_IDirect3DViewport3, refiid) )
    {
        IDirect3DViewport3_AddRef(iface);
        *obj = iface;
        TRACE("  Creating IDirect3DViewport1/2/3 interface %p\n", *obj);
        return S_OK;
    }

    FIXME("(%p): interface for IID %s NOT found!\n", This, debugstr_guid(refiid));
    return E_NOINTERFACE;
}

/*****************************************************************************
 * IDirect3DViewport3::AddRef
 *
 * Increases the refcount.
 *
 * Returns:
 *  The new refcount
 *
 *****************************************************************************/
static ULONG WINAPI
IDirect3DViewportImpl_AddRef(IDirect3DViewport3 *iface)
{
    ICOM_THIS_FROM(IDirect3DViewportImpl, IDirect3DViewport3, iface);
    ULONG ref = InterlockedIncrement(&This->ref);

    TRACE("(%p)->() incrementing from %lu.\n", This, ref - 1);

    return ref;
}

/*****************************************************************************
 * IDirect3DViewport3::Release
 *
 * Reduces the refcount. If it falls to 0, the interface is released
 *
 * Returns:
 *  The new refcount
 *
 *****************************************************************************/
static ULONG WINAPI
IDirect3DViewportImpl_Release(IDirect3DViewport3 *iface)
{
    ICOM_THIS_FROM(IDirect3DViewportImpl, IDirect3DViewport3, iface);
    ULONG ref = InterlockedDecrement(&This->ref);

    TRACE("(%p)->() decrementing from %lu.\n", This, ref + 1);

    if (!ref)
    {
        HeapFree(GetProcessHeap(), 0, This);
    }
    return ref;
}

/*****************************************************************************
 * IDirect3DViewport Methods.
 *****************************************************************************/

/*****************************************************************************
 * IDirect3DViewport3::Initialize
 *
 * No-op initialization.
 *
 * Params:
 *  Direct3D: The direct3D device this viewport is assigned to
 *
 * Returns:
 *  DDERR_ALREADYINITIALIZED
 *
 *****************************************************************************/
static HRESULT WINAPI
IDirect3DViewportImpl_Initialize(IDirect3DViewport3 *iface,
                                 IDirect3D *Direct3D)
{
    TRACE("(%p)->(%p) no-op...\n", iface, Direct3D);
    return DDERR_ALREADYINITIALIZED;
}

/*****************************************************************************
 * IDirect3DViewport3::GetViewport
 *
 * Returns the viewport data assigned to this viewport interface
 *
 * Params:
 *  Data: Address to store the data
 *
 * Returns:
 *  D3D_OK on success
 *  DDERR_INVALIDPARAMS if Data is NULL
 *
 *****************************************************************************/
static HRESULT WINAPI
IDirect3DViewportImpl_GetViewport(IDirect3DViewport3 *iface,
                                  D3DVIEWPORT *Data)
{
    ICOM_THIS_FROM(IDirect3DViewportImpl, IDirect3DViewport3, iface);
    DWORD Size;
    TRACE("(%p)->(%p)\n", This, Data);

    if(!Data)
        return DDERR_INVALIDPARAMS;

    if (This->use_vp2 != 0)
    {
        ERR("  Requesting to get a D3DVIEWPORT struct where a D3DVIEWPORT2 was set !\n");
        return DDERR_INVALIDPARAMS;
    }
    Size = Data->dwSize;
    memset(Data, 0, Size);
    memcpy(Data, &(This->viewports.vp1), Size);

    if (TRACE_ON(d3d7))
    {
        TRACE("  returning D3DVIEWPORT :\n");
        _dump_D3DVIEWPORT(Data);
    }

    return D3D_OK;
}

/*****************************************************************************
 * IDirect3DViewport3::SetViewport
 *
 * Sets the viewport information for this interface
 *
 * Params:
 *  Data: Viewport to set
 *
 * Returns:
 *  D3D_OK on succes
 *  DDERR_INVALIDPARAMS if Data is NULL
 *
 *****************************************************************************/
HRESULT WINAPI
IDirect3DViewportImpl_SetViewport(IDirect3DViewport3 *iface,
                                  D3DVIEWPORT *Data)
{
    ICOM_THIS_FROM(IDirect3DViewportImpl, IDirect3DViewport3, iface);
    LPDIRECT3DVIEWPORT3 current_viewport;
    TRACE("(%p)->(%p)\n", This, Data);

    if (TRACE_ON(d3d7))
    {
        TRACE("  getting D3DVIEWPORT :\n");
        _dump_D3DVIEWPORT(Data);
    }

    This->use_vp2 = 0;
    memset(&(This->viewports.vp1), 0, sizeof(This->viewports.vp1));
    memcpy(&(This->viewports.vp1), Data, Data->dwSize);

    /* Tests on two games show that these values are never used properly so override
       them with proper ones :-)
    */
    This->viewports.vp1.dvMinZ = 0.0;
    This->viewports.vp1.dvMaxZ = 1.0;

    if (This->active_device)
    {
      IDirect3DDevice3_GetCurrentViewport(ICOM_INTERFACE(This->active_device, IDirect3DDevice3),
                                          &current_viewport);
      if (ICOM_OBJECT(IDirect3DViewportImpl, IDirect3DViewport3, current_viewport) == This)
          This->activate(This);
      IDirect3DViewport3_Release(current_viewport);
    }

    return D3D_OK;
}

/*****************************************************************************
 * IDirect3DViewport3::TransformVertices
 *
 * Transforms vertices by the transformation matrix.
 *
 * Params:
 *  VertexCount: The number of vertices to be transformed
 *  Data: Pointer to the vertex data
 *  Flags: D3DTRANSFORM_CLIPPED or D3DTRANSFORM_UNCLIPPED
 *  OffScreen: Is set to nonzero if all vertices are off-screen
 *
 * Returns:
 *  D3D_OK because it's a stub
 *
 *****************************************************************************/
static HRESULT WINAPI
IDirect3DViewportImpl_TransformVertices(IDirect3DViewport3 *iface,
                                        DWORD VertexCount,
                                        D3DTRANSFORMDATA *Data,
                                        DWORD Flags,
                                        DWORD *OffScreen)
{
    ICOM_THIS_FROM(IDirect3DViewportImpl, IDirect3DViewport3, iface);
    FIXME("(%p)->(%08lx,%p,%08lx,%p): stub!\n", This, VertexCount, Data, Flags, OffScreen);

    if (OffScreen)
        *OffScreen = 0;
    return D3D_OK;
}

/*****************************************************************************
 * IDirect3DViewport3::LightElements
 *
 * The DirectX 5.0 sdk says that it's not implemented
 *
 * Params:
 *  ?
 *
 * Returns:
 *  DDERR_UNSUPPORTED
 *
 *****************************************************************************/
static HRESULT WINAPI
IDirect3DViewportImpl_LightElements(IDirect3DViewport3 *iface,
                                    DWORD ElementCount,
                                    LPD3DLIGHTDATA Data)
{
    ICOM_THIS_FROM(IDirect3DViewportImpl, IDirect3DViewport3, iface);
    TRACE("(%p)->(%08lx,%p): not implemented!\n", This, ElementCount, Data);

    return DDERR_UNSUPPORTED;
}

/*****************************************************************************
 * IDirect3DViewport3::SetBackground
 *
 * Sets tje background material
 *
 * Params:
 *  Mat: Handle from a IDirect3DMaterial interface
 *
 * Returns:
 *  D3D_OK on success
 *
 *****************************************************************************/
static HRESULT WINAPI
IDirect3DViewportImpl_SetBackground(IDirect3DViewport3 *iface,
                                    D3DMATERIALHANDLE Mat)
{
    ICOM_THIS_FROM(IDirect3DViewportImpl, IDirect3DViewport3, iface);
    TRACE("(%p)->(%08lx)\n", This, (DWORD) Mat);

    This->background = (IDirect3DMaterialImpl *) Mat;
    TRACE(" setting background color : %f %f %f %f\n",
          This->background->mat.diffuse.r,
          This->background->mat.diffuse.g,
          This->background->mat.diffuse.b,
          This->background->mat.diffuse.a);

    return D3D_OK;
}

/*****************************************************************************
 * IDirect3DViewport3::GetBackground
 *
 * Returns the material handle assigned to the background of the viewport
 *
 * Params:
 *  Mat: Address to store the handle
 *  Valid: is set to FALSE if no background is set, TRUE if one is set
 *
 * Returns:
 *  D3D_OK, because it's a stub
 *  (DDERR_INVALIDPARAMS if Mat or Valid is NULL)
 *
 *****************************************************************************/
static HRESULT WINAPI
IDirect3DViewportImpl_GetBackground(IDirect3DViewport3 *iface,
                                    D3DMATERIALHANDLE *Mat,
                                    BOOL *Valid)
{
    FIXME("(%p)->(%p,%p): stub!\n", iface, Mat, Valid);

    return D3D_OK;
}

/*****************************************************************************
 * IDirect3DViewport3::SetBackgroundDepth
 *
 * Sets a surface that represents the background depth. It's contents are
 * used to set the depth buffer in IDirect3DViewport3::Clear
 *
 * Params:
 *  DDSurface: Surface to set
 *
 * Returns: D3D_OK, because it's a stub
 *
 *****************************************************************************/
static HRESULT WINAPI
IDirect3DViewportImpl_SetBackgroundDepth(IDirect3DViewport3 *iface,
                                         IDirectDrawSurface *DDSurface)
{
    ICOM_THIS_FROM(IDirect3DViewportImpl, IDirect3DViewport3, iface);
    IDirectDrawSurfaceImpl *surf = ICOM_OBJECT(IDirectDrawSurfaceImpl, IDirectDrawSurface3, DDSurface);
    FIXME("(%p)->(%p): stub!\n", This, surf);

    return D3D_OK;
}

/*****************************************************************************
 * IDirect3DViewport3::GetBackgroundDepth
 *
 * Returns the surface that represents the depth field
 *
 * Params:
 *  DDSurface: Address to store the interface pointer
 *  Valid: Set to TRUE if a depth is asigned, FALSE otherwise
 *
 * Returns:
 *  D3D_OK, because it's a stub
 *  (DDERR_INVALIDPARAMS if DDSurface of Valid is NULL)
 *
 *****************************************************************************/
static HRESULT WINAPI
IDirect3DViewportImpl_GetBackgroundDepth(IDirect3DViewport3 *iface,
                                         IDirectDrawSurface **DDSurface,
                                         LPBOOL Valid)
{
    ICOM_THIS_FROM(IDirect3DViewportImpl, IDirect3DViewport3, iface);
    FIXME("(%p)->(%p,%p): stub!\n", This, DDSurface, Valid);

    return D3D_OK;
}

/*****************************************************************************
 * IDirect3DViewport3::Clear
 *
 * Clears the render target and / or the z buffer
 *
 * Params:
 *  Count: The amount of rectangles to clear. If 0, the whole buffer is
 *         cleared
 *  Rects: Pointer to the array of rectangles. If NULL, Count must be 0
 *  Flags: D3DCLEAR_ZBUFFER and / or D3DCLEAR_TARGET
 *
 * Returns:
 *  D3D_OK on success
 *  D3DERR_VIEWPORTHASNODEVICE if there's no active device
 *  The return value of IDirect3DDevice7::Clear
 *
 *****************************************************************************/
static HRESULT WINAPI
IDirect3DViewportImpl_Clear(IDirect3DViewport3 *iface,
                            DWORD Count, 
                            D3DRECT *Rects,
                            DWORD Flags)
{
    ICOM_THIS_FROM(IDirect3DViewportImpl, IDirect3DViewport3, iface);
    DWORD color = 0x00000000;

    TRACE("(%p)->(%08lx,%p,%08lx)\n", This, Count, Rects, Flags);
    if (This->active_device == NULL)
    {
        ERR(" Trying to clear a viewport not attached to a device !\n");
        return D3DERR_VIEWPORTHASNODEVICE;
    }

    if (Flags & D3DCLEAR_TARGET)
    {
        if (This->background == NULL)
        {
            ERR(" Trying to clear the color buffer without background material !\n");
        }
        else
        {
            color =
              ((int) ((This->background->mat.diffuse.r) * 255) << 16) |
              ((int) ((This->background->mat.diffuse.g) * 255) <<  8) |
              ((int) ((This->background->mat.diffuse.b) * 255) <<  0) |
              ((int) ((This->background->mat.diffuse.a) * 255) << 24);
        }
    }

    return IDirect3DDevice7_Clear(ICOM_INTERFACE(This->active_device, IDirect3DDevice7),
                                  Count,
                                  Rects,
                                  Flags & (D3DCLEAR_ZBUFFER | D3DCLEAR_TARGET),
                                  color,
                                  1.0,
                                  0x00000000);
}

/*****************************************************************************
 * IDirect3DViewport3::AddLight
 *
 * Adds an light to the viewport
 *
 * Params:
 *  Direct3DLight: Interface of the light to add
 *
 * Returns:
 *  D3D_OK on success
 *  DDERR_INVALIDPARAMS if Direct3DLight is NULL
 *  DDERR_INVALIDPARAMS if there are 8 lights or more
 *
 *****************************************************************************/
static HRESULT WINAPI
IDirect3DViewportImpl_AddLight(IDirect3DViewport3 *iface,
                               IDirect3DLight *Direct3DLight)
{
    ICOM_THIS_FROM(IDirect3DViewportImpl, IDirect3DViewport3, iface);
    IDirect3DLightImpl *Direct3DLightImpl = ICOM_OBJECT(IDirect3DLightImpl, IDirect3DLight, Direct3DLight);
    DWORD i = 0;
    DWORD map = This->map_lights;

    TRACE("(%p)->(%p)\n", This, Direct3DLightImpl);

    if(!Direct3DLight)
        return DDERR_INVALIDPARAMS;

    if (This->num_lights >= 8)
        return DDERR_INVALIDPARAMS;

    /* Find a light number and update both light and viewports objects accordingly */
    while(map&1)
    {
        map>>=1;
        i++;
    }
    Direct3DLightImpl->dwLightIndex = i;
    This->num_lights++;
    This->map_lights |= 1<<i;

    /* Add the light in the 'linked' chain */
    Direct3DLightImpl->next = This->lights;
    This->lights = Direct3DLightImpl;

    /* Attach the light to the viewport */
    Direct3DLightImpl->active_viewport = This;

    /* If active, activate the light */
    if (This->active_device != NULL)
    {
        Direct3DLightImpl->activate(Direct3DLightImpl);
    }

    return D3D_OK;
}

/*****************************************************************************
 * IDirect3DViewport3::DeleteLight
 *
 * Deletes a light from the viewports' light list
 *
 * Params:
 *  Direct3DLight: Light to delete
 *
 * Returns:
 *  D3D_OK on success
 *  DDERR_INVALIDPARAMS if the light wasn't found
 *
 *****************************************************************************/
static HRESULT WINAPI
IDirect3DViewportImpl_DeleteLight(IDirect3DViewport3 *iface,
                                  IDirect3DLight *Direct3DLight)
{
    ICOM_THIS_FROM(IDirect3DViewportImpl, IDirect3DViewport3, iface);
    IDirect3DLightImpl *Direct3DLightImpl = ICOM_OBJECT(IDirect3DLightImpl, IDirect3DLight, Direct3DLight);
    IDirect3DLightImpl *cur_light, *prev_light = NULL;

    TRACE("(%p)->(%p)\n", This, Direct3DLightImpl);
    cur_light = This->lights;
    while (cur_light != NULL)
    {
        if (cur_light == Direct3DLightImpl)
        {
            Direct3DLightImpl->desactivate(Direct3DLightImpl);
            if (prev_light == NULL) This->lights = cur_light->next;
            else prev_light->next = cur_light->next;
            /* Detach the light to the viewport */
            cur_light->active_viewport = NULL;
            This->num_lights--;
            This->map_lights &= ~(1<<Direct3DLightImpl->dwLightIndex);
            return D3D_OK;
        }
        prev_light = cur_light;
        cur_light = cur_light->next;
    }
    return DDERR_INVALIDPARAMS;
}

/*****************************************************************************
 * IDirect3DViewport::NextLight
 *
 * Enumerates the lights associated with the viewport
 *
 * Params:
 *  lpDirect3DLight: Light to start with
 *  lplpDirect3DLight: Address to store the successor to
 *
 * Returns:
 *  D3D_OK, because it's a stub
 *
 *****************************************************************************/
static HRESULT WINAPI
IDirect3DViewportImpl_NextLight(IDirect3DViewport3 *iface,
                                IDirect3DLight *lpDirect3DLight,
                                IDirect3DLight **lplpDirect3DLight,
                                DWORD Flags)
{
    FIXME("(%p)->(%p,%p,%08lx): stub!\n", iface, lpDirect3DLight, lplpDirect3DLight, Flags);

    return D3D_OK;
}

/*****************************************************************************
 * IDirect3DViewport2 Methods.
 *****************************************************************************/

/*****************************************************************************
 * IDirect3DViewport3::GetViewport2
 *
 * Returns the currently set viewport in a D3DVIEWPORT2 structure.
 * Simmilar to IDirect3DViewport3::GetViewport
 *
 * Params:
 *  Data: Pointer to the structure to fill
 *
 * Returns:
 *  D3D_OK on success
 *  DDERR_INVALIDPARAMS if the viewport was set with
 *                      IDirect3DViewport3::SetViewport
 *  DDERR_INVALIDPARAMS if Data is NULL
 *
 *****************************************************************************/
static HRESULT WINAPI
IDirect3DViewportImpl_GetViewport2(IDirect3DViewport3 *iface,
                                   D3DVIEWPORT2 *Data)
{
    ICOM_THIS_FROM(IDirect3DViewportImpl, IDirect3DViewport3, iface);
    DWORD Size;
    TRACE("(%p)->(%p)\n", This, Data);

    if(!Data)
        return DDERR_INVALIDPARAMS;

    if (This->use_vp2 != 1)
    {
        ERR("  Requesting to get a D3DVIEWPORT2 struct where a D3DVIEWPORT was set !\n");
        return DDERR_INVALIDPARAMS;
    }

    Size = Data->dwSize;
    memset(Data, 0, Size);
    memcpy(Data, &(This->viewports.vp2), Size);

    if (TRACE_ON(d3d7))
    {
        TRACE("  returning D3DVIEWPORT2 :\n");
        _dump_D3DVIEWPORT2(Data);
    }

    return DD_OK;
}

/*****************************************************************************
 * IDirect3DViewport3::SetViewport2
 *
 * Sets the viewport from a D3DVIEWPORT2 structure
 *
 * Params:
 *  Data: Viewport to set
 *
 * Returns:
 *  D3D_OK on success
 *
 *****************************************************************************/
static HRESULT WINAPI
IDirect3DViewportImpl_SetViewport2(IDirect3DViewport3 *iface,
                                   D3DVIEWPORT2 *Data)
{
    ICOM_THIS_FROM(IDirect3DViewportImpl, IDirect3DViewport3, iface);
    IDirect3DViewport3 *current_viewport;
    TRACE("(%p)->(%p)\n", This, Data);

    if (TRACE_ON(d3d7))
    {
        TRACE("  getting D3DVIEWPORT2 :\n");
        _dump_D3DVIEWPORT2(Data);
    }

    This->use_vp2 = 1;
    memset(&(This->viewports.vp2), 0, sizeof(This->viewports.vp2));
    memcpy(&(This->viewports.vp2), Data, Data->dwSize);

    if (This->active_device)
    {
      IDirect3DDevice3_GetCurrentViewport( ICOM_INTERFACE(This->active_device, IDirect3DDevice3), &current_viewport);
      if ( ICOM_OBJECT(IDirect3DViewportImpl, IDirect3DViewport3, current_viewport) == This)
        This->activate(This);
      IDirect3DViewport3_Release(current_viewport);
    }

    return DD_OK;
}

/*****************************************************************************
 * IDirect3DViewport3 Methods.
 *****************************************************************************/

/*****************************************************************************
 * IDirect3DViewport3::SetBackgroundDepth2
 *
 * Sets a IDirectDrawSurface4 surface as the background depth surface
 *
 * Params:
 *  lpDDS: Surface to set
 *
 * Returns:
 *  D3D_OK, because it's stub
 *
 *****************************************************************************/
static HRESULT WINAPI
IDirect3DViewportImpl_SetBackgroundDepth2(IDirect3DViewport3 *iface,
                                          IDirectDrawSurface4 *DDS)
{
    ICOM_THIS_FROM(IDirect3DViewportImpl, IDirect3DViewport3, iface);
    IDirectDrawSurfaceImpl *surf = ICOM_OBJECT(IDirectDrawSurfaceImpl, IDirectDrawSurface7, DDS);
    FIXME("(%p)->(%p): stub!\n", This, surf);

    return D3D_OK;
}

/*****************************************************************************
 * IDirect3DViewport3::GetBackgroundDepth2
 *
 * Returns the IDirect3DSurface4 interface to the background depth surface
 *
 * Params:
 *  DDS: Address to store the interface pointer at
 *  Valid: Set to true if a surface is assigned
 *
 * Returns:
 *  D3D_OK because it's a stub
 *
 *****************************************************************************/
static HRESULT WINAPI
IDirect3DViewportImpl_GetBackgroundDepth2(IDirect3DViewport3 *iface,
                                          IDirectDrawSurface4 **DDS,
                                          BOOL *Valid)
{
    ICOM_THIS_FROM(IDirect3DViewportImpl, IDirect3DViewport3, iface);
    FIXME("(%p)->(%p,%p): stub!\n", This, DDS, Valid);

    return D3D_OK;
}

/*****************************************************************************
 * IDirect3DViewport3::Clear2
 *
 * Another clearing method
 *
 * Params:
 *  Count: Number of rectangles to clear
 *  Rects: Rectangle array to clear
 *  Flags: Some flags :)
 *  Color: Color to fill the render target with
 *  Z: Value to fill the depth buffer with
 *  Stencil: Value to fill the stencil bits with
 *
 * Returns:
 *
 *****************************************************************************/
static HRESULT WINAPI
IDirect3DViewportImpl_Clear2(IDirect3DViewport3 *iface,
                             DWORD Count,
                             LPD3DRECT Rects,
                             DWORD Flags,
                             DWORD Color,
                             D3DVALUE Z,
                             DWORD Stencil)
{
    ICOM_THIS_FROM(IDirect3DViewportImpl, IDirect3DViewport3, iface);
    TRACE("(%p)->(%08lx,%p,%08lx,%08lx,%f,%08lx): Relaying to D3D7\n", This, Count, Rects, Flags, Color, Z, Stencil);

    if (This->active_device == NULL)
    {
        ERR(" Trying to clear a viewport not attached to a device !\n");
        return D3DERR_VIEWPORTHASNODEVICE;
    }
    return IDirect3DDevice7_Clear(ICOM_INTERFACE(This->active_device, IDirect3DDevice7),
                                  Count,
                                  Rects,
                                  Flags,
                                  Color,
                                  Z,
                                  Stencil);
}

/*****************************************************************************
 * The VTable
 *****************************************************************************/

const IDirect3DViewport3Vtbl IDirect3DViewport3_Vtbl =
{
    /*** IUnknown Methods ***/
    IDirect3DViewportImpl_QueryInterface,
    IDirect3DViewportImpl_AddRef,
    IDirect3DViewportImpl_Release,
    /*** IDirect3DViewport Methods */
    IDirect3DViewportImpl_Initialize,
    IDirect3DViewportImpl_GetViewport,
    IDirect3DViewportImpl_SetViewport,
    IDirect3DViewportImpl_TransformVertices,
    IDirect3DViewportImpl_LightElements,
    IDirect3DViewportImpl_SetBackground,
    IDirect3DViewportImpl_GetBackground,
    IDirect3DViewportImpl_SetBackgroundDepth,
    IDirect3DViewportImpl_GetBackgroundDepth,
    IDirect3DViewportImpl_Clear,
    IDirect3DViewportImpl_AddLight,
    IDirect3DViewportImpl_DeleteLight,
    IDirect3DViewportImpl_NextLight,
    /*** IDirect3DViewport2 Methods ***/
    IDirect3DViewportImpl_GetViewport2,
    IDirect3DViewportImpl_SetViewport2,
    /*** IDirect3DViewport3 Methods ***/
    IDirect3DViewportImpl_SetBackgroundDepth2,
    IDirect3DViewportImpl_GetBackgroundDepth2,
    IDirect3DViewportImpl_Clear2,
};
